package gov.va.genisis2.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.common.enums.UserStatusEnum;
import gov.va.genisis2.converter.StudyApprovalConverter;
import gov.va.genisis2.dao.ILdapStudyApprovalDAO;
import gov.va.genisis2.dao.IRefreshHistoryDao;
import gov.va.genisis2.dao.IStudyApprovalDao;
import gov.va.genisis2.dao.IUserManagementDao;
import gov.va.genisis2.dto.StudyApprovalDTO;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.exceptions.GenisisServiceException;
import gov.va.genisis2.model.RefreshHistory;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.service.IStudyApprovalService;
import gov.va.genisis2.util.DateUtil;
import gov.va.genisis2.vo.LdapStudyApproval;
import gov.va.genisis2.vo.LdapStudyApprovalGroup;

/**
 * @author 586338
 * @author Getaneh Kassahun
 *
 */
@Service
public class StudyApprovalService implements IStudyApprovalService {

	private static final Logger LOGGER = LoggerFactory.getLogger(StudyApprovalService.class);
	private static final Logger LOGGERlDAPREFRESH = LoggerFactory.getLogger("gov.va.genisis2.schedule");

	private static final String LDAP_ROLE_GROUP_DELIMETER = ", ";

	@Autowired
	private ILdapStudyApprovalDAO ldapStudyApprovalDAO;

	@Autowired
	private StudyApprovalConverter studyApprovalConverter;

	@Autowired
	private IRefreshHistoryDao refreshHistoryDao;

	@Autowired
	private IUserManagementDao userManagementDao;

	@Autowired
	private IStudyApprovalDao studyApprovalDao;

	/**
	 * /** This method is used to createStudyApproval.
	 * 
	 * @param studyApproval
	 *            The studyApproval.
	 * @return int This returns studyApproval id.
	 */
	@Override
	public int createStudyApproval(StudyApproval studyApproval) throws GenisisServiceException {
		try {
			return studyApprovalDao.createStudyApproval(studyApproval);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on createStudyApproval", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to updateStudyApproval.
	 * 
	 * @param studyApproval
	 *            The studyApproval.
	 * @return int This returns studyApproval id.
	 */
	@Override
	public int updateStudyApproval(StudyApproval studyApproval) throws GenisisServiceException {
		try {
			return studyApprovalDao.updateStudyApproval(studyApproval);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on updateStudyApproval", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getStudyApprovalsByID.
	 * 
	 * @param id
	 *            The id.
	 * @return StudyApproval This returns StudyApproval.
	 */
	@Override
	public StudyApproval getStudyApprovalsByID(int id) throws GenisisServiceException {
		try {
			return studyApprovalDao.getStudyApprovalsByID(id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querring  getStudyApprovalsByID", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getStudyApprovalsByUID.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovalsByUID(String uid) throws GenisisServiceException {
		try {
			return studyApprovalDao.getStudyApprovalsByUID(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querring  getStudyApprovalsByUID", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getStudyApprovals.
	 * 
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovals() throws GenisisServiceException {
		try {
			return studyApprovalDao.getStudyApprovals();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querring  getStudyApprovals", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public RefreshHistory updateStudyApprovalRefreshHistory(RefreshHistory refreshHistory) throws GenisisServiceException {
		try {
			return refreshHistoryDao.updateUserRefreshHistory(refreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying updateUserRefreshHistory.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}
//ok
	@Override
	public void refreshStudyApproval() throws GenisisServiceException {
		List<LdapStudyApprovalGroup> ldapStudyApprovalGroups;
		List<StudyApproval> daoStudyApprovals;
		List<String> updatedStudyApprovalDiscription = new ArrayList<>();
		try {
			ldapStudyApprovalGroups = ldapStudyApprovalDAO.getAllLdapStudyApproval();
			daoStudyApprovals = studyApprovalDao.getStudyApprovals();
			
			StringJoiner sjLdapGroups = new StringJoiner(LDAP_ROLE_GROUP_DELIMETER);
			// populate studyApproval details for each ldapStudyApprovalGroups
			populateStudyApprovalsForLdap(ldapStudyApprovalGroups,sjLdapGroups);
			
			// ignore users with multiple roles  (TBD)
			Set<String> duplicateUsers = ignoreUsersWithMultipleRoles(ldapStudyApprovalGroups, daoStudyApprovals);
			
			// create a user map
			Map<String, StudyApproval> studyApprovalMap = createUserMap(daoStudyApprovals);
			
			// add/update study approval details in StudyApproval table with  LDAP
			for (LdapStudyApprovalGroup ldapSAGroup : ldapStudyApprovalGroups) {
				if (null != ldapSAGroup) {
					compareStudyApproval(ldapSAGroup, studyApprovalMap, updatedStudyApprovalDiscription);
				}
			}
			
			// Find Study Approvals to be disabled in database
			if (sjLdapGroups != null && studyApprovalMap !=null) {
				listStudyApprovalsInDatabaseDisable(sjLdapGroups, studyApprovalMap, updatedStudyApprovalDiscription);
			}
			
			// TBD
			if (!duplicateUsers.isEmpty()) {
				updatedStudyApprovalDiscription.addAll(duplicateUsers);
			}
			
			// update user status to 'DISABLE'
			updateStudyApprovalStatus(updatedStudyApprovalDiscription);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getAllUsers from LDAP", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}
//ok
	private void compareStudyApproval(LdapStudyApprovalGroup ldapStudyApprovalGroup, Map<String, StudyApproval> studyApprovalMap, List<String> updatedStudyApprovalNames) throws GenisisDAOException {
		List<StudyApproval> ldapStudyApprovals = ldapStudyApprovalGroup.getLdapStudyApprovals();
		StudyApproval studyApproval;
		StudyApproval persistentStudyApproval;
		StudyApprovalDTO studyApprovalDTO = null;
		if ((null == ldapStudyApprovals || ldapStudyApprovals.isEmpty()) && StringUtils.isBlank(ldapStudyApprovalGroup.getGroupName())) {
			return;
		}
		
		if ((null == ldapStudyApprovals || ldapStudyApprovals.isEmpty()) && !StringUtils.isBlank(ldapStudyApprovalGroup.getGroupName())) {
			StudyApproval object = new StudyApproval();
			object.setTitle(ldapStudyApprovalGroup.getGroupName());
			object.setDescription(ldapStudyApprovalGroup.getGroupName());
			object.setActive((byte) UserStatusEnum.ACTIVE.getId());
			object.setActiveDate(DateUtil.getTodaysDate());
			ldapStudyApprovals.add(object);
		}
		
		for (StudyApproval ldapSa : ldapStudyApprovals) {
			if (null == ldapSa) {
				LOGGER.error("ldapSa object is null");
				continue;
			}
			if (StringUtils.equalsIgnoreCase(ldapSa.getTitle(), "Operations")) {
				LOGGER.error("Ignore Operations in StudyApproval");
				continue;
			}
			// no study approval in DB, every LDAP study approval is a new entry
			if (null == studyApprovalMap || studyApprovalMap.isEmpty()) {
				studyApprovalDTO = studyApprovalConverter.convert(ldapSa);
				studyApproval = studyApprovalConverter.convert(studyApprovalDTO);
			} else {
				studyApproval = studyApprovalMap.get(ldapSa.getDescription());
				// new study approval
				if (null == studyApproval) {
					ldapSa.setActive((byte) UserStatusEnum.ACTIVE.getId());
					ldapSa.setActiveDate(DateUtil.getTodaysDate());
					studyApprovalDTO = studyApprovalConverter.convert(ldapSa);
					studyApproval = studyApprovalConverter.convert(studyApprovalDTO);
				}
			}
			// persist user
			if (studyApproval != null) {
				persistentStudyApproval = studyApprovalDao.saveOrUpdate(studyApproval);
			}
		}
	}

	//ok
	private void updateStudyApprovalStatus(List<String> updatedStudyApprovalNames) throws GenisisServiceException {
		boolean flag = null == updatedStudyApprovalNames || updatedStudyApprovalNames.isEmpty();
		if (flag) {
			return;
		}
		try {
			studyApprovalDao.updateStudyApprovalStatus(updatedStudyApprovalNames);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.STUDYAPPROVAL_DAO_EXP_UPDATESTATUS.getErrorMessage(), ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	// ok
	private Map<String, StudyApproval> createUserMap(List<StudyApproval> daoStudyApprovals) {
		Map<String, StudyApproval> userMap;
		if (null == daoStudyApprovals || daoStudyApprovals.isEmpty()) {
			return null;
		} else {
			userMap = new HashMap<>();
			for (StudyApproval studyApproval : daoStudyApprovals) {
				String discription = studyApproval.getDescription();
				String upperCaseDiscription = (null == discription || discription.isEmpty()) ? StringUtils.EMPTY : discription.trim().toUpperCase();
				studyApproval.setDescription(upperCaseDiscription);
				userMap.put(upperCaseDiscription, studyApproval);
			}
		}
		return userMap;
	}

	/**
	 * To ignore Users With Multiple Roles
	 * 
	 */
	private Set<String> ignoreUsersWithMultipleRoles(List<LdapStudyApprovalGroup> ldapStudyApprovalGroups, List<StudyApproval> daoStudyApprovals) {
		List<String> allStudyApprovalList = new ArrayList<>();

		// Get a List of studyApproval from all groups
		for (LdapStudyApprovalGroup ldapStudyApprovalGroup : ldapStudyApprovalGroups) {
			String groupName = ldapStudyApprovalGroup.getGroupName();
			if (!StringUtils.isBlank(groupName)) {
				allStudyApprovalList.add(groupName);
			}
		}
		if (!findDuplicates(allStudyApprovalList).isEmpty()) {
			removeSaFromLdapSaGroupAndSAs(findDuplicates(allStudyApprovalList), ldapStudyApprovalGroups, daoStudyApprovals);
		}
		return findDuplicates(allStudyApprovalList);
	}

	/**
	 * To find Duplicate study Approval
	 * 
	 * @param groupsuserList
	 * @return string
	 * 
	 */
	private Set<String> findDuplicates(List<String> allStudyApprovalList) {
		final Set<String> duplicateStudyApproval = new HashSet<>();

		final Set<String> set1 = new HashSet<>();
		for (String studyApproval : allStudyApprovalList) {
			if (!set1.add(studyApproval)) {
				duplicateStudyApproval.add(studyApproval);
			}
		}
		return duplicateStudyApproval;
	}

	private void removeSaFromLdapSaGroupAndSAs(Set<String> duplicateStudyApprovals, List<LdapStudyApprovalGroup> ldapStudyApprovalGroups, List<StudyApproval> daoStudyApprovals) {
		Set<String> logOnlyOnce = new HashSet<>();
		Map<String, String> ignoredStudyApproval = new HashMap<>();
		if (null != duplicateStudyApprovals) {
			for (LdapStudyApprovalGroup ldapGroupStudyApproval : ldapStudyApprovalGroups) {
				// Remove User with multiple Role from LdapUserGroup object
				removeUserFromLdapUserGroup(ldapGroupStudyApproval, duplicateStudyApprovals, logOnlyOnce, ignoredStudyApproval);

				// Remove StudyAproval with multiple Role from users object
				removeStudyApprovalfromDaoStudyApproval(duplicateStudyApprovals, daoStudyApprovals);
			}
		}
		// Log users could not be added to the database
		if (ignoredStudyApproval.size() > 0) {
			Iterator<Entry<String, String>> it = ignoredStudyApproval.entrySet().iterator();
			while (it.hasNext()) {
				@SuppressWarnings("rawtypes")
				Map.Entry pair = (Map.Entry) it.next();
				LOGGERlDAPREFRESH.info("The user " + pair.getKey() + " has multiple roles defined in LDAP server. This user will not be added to the database. Roles: " + pair.getValue());
			}
		}
	}

	/**
	 * To remove studyApproval From Ldap studyApproval Group
	 * 
	 * @param ldapGroupStudyApproval
	 * @param duplicateStudyApproval
	 * 
	 * 
	 */
	private void removeUserFromLdapUserGroup(LdapStudyApprovalGroup ldapGroupStudyApproval, Set<String> duplicateStudyApprovals, Set<String> logOnlyOnce, Map<String, String> ignoredStudyApproval) {
	/*	if (ldapGroupStudyApproval != null) {

			// Remove studyApproval with multiple Role from
			// LdapStudyApprovalGroup
			for (Iterator<LdapStudyApproval> iterator = (ldapGroupStudyApproval.getldapStudyApprovals()).iterator(); iterator.hasNext();) {
				LdapStudyApproval ldapStudyApproval = iterator.next();
				if (ldapStudyApproval != null && !duplicateStudyApprovals.toString().isEmpty() && StringUtils.contains(duplicateStudyApprovals.toString(), ldapStudyApproval.getDescription())) {
					String mapValue = ignoredStudyApproval.get(ldapStudyApproval.getDescription());
					if (StringUtils.isBlank(mapValue)) {
						ignoredStudyApproval.put(ldapStudyApproval.getDescription(), ldapGroupStudyApproval.getGroupName());
					} else {
						ignoredStudyApproval.put(ldapStudyApproval.getDescription(), mapValue + LDAP_ROLE_GROUP_DELIMETER + ldapGroupStudyApproval.getGroupName());
					}
					iterator.remove();
				}
			}
		}*/
	}

	/**
	 * To remove StudyApproval from daoStudyApproval
	 * 
	 * @param duplicateUsers
	 * @param users
	 * 
	 * 
	 */
	private void removeStudyApprovalfromDaoStudyApproval(Set<String> duplicateStudyApprovals, List<StudyApproval> daoStudyApprovals) {
		// Remove User with multiple Role from users object
		for (Iterator<StudyApproval> iterator = daoStudyApprovals.iterator(); iterator.hasNext();) {
			StudyApproval ldapStudyApproval = iterator.next();
			if (ldapStudyApproval != null && !duplicateStudyApprovals.toString().isEmpty() && StringUtils.contains(duplicateStudyApprovals.toString(), ldapStudyApproval.getDescription())) {
				// iterator.remove();
			}
		}
	}

	/**
	 * @param studyApprovalDao
	 *            the studyApprovalDao to set
	 */
	public void setStudyApprovalDao(IStudyApprovalDao studyApprovalDao) {
		this.studyApprovalDao = studyApprovalDao;
	}

	
	//ok
	public  List<StudyApproval> getStudyApprovalsByTitle(String title) throws GenisisServiceException {
		try {
			return studyApprovalDao.getStudyApprovalsByTitle(title);
			} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getStudyApprovalsByTitle.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	//ok
	private void listStudyApprovalsInDatabaseDisable(StringJoiner saFromLdap, Map<String, StudyApproval> studyApprovalMap, List<String> updatedStudyApprovalNames) {
		// No Study Approval in ldap but available in db, disbale the study
		// approval
		Boolean saPresentDb = false;
		Iterator it = studyApprovalMap.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry pair = (Map.Entry) it.next();
			String databaseSa = StringUtils.upperCase(pair.getKey().toString());
			if (StringUtils.countMatches(saFromLdap.toString(), databaseSa) > 0) {
				saPresentDb = true;
				continue;
			} else {
				updatedStudyApprovalNames.add(databaseSa);
			}
		}
	}
	
	//ok
	private void populateStudyApprovalsForLdap(List<LdapStudyApprovalGroup> ldapStudyApprovalGroups,StringJoiner sjLdapGroups) {
		for (LdapStudyApprovalGroup ldapStudyApprovalGroup : ldapStudyApprovalGroups) {
			String studyApprovalTitle = ldapStudyApprovalGroup.getGroupName();
			sjLdapGroups.add(StringUtils.upperCase(studyApprovalTitle));
			List<StudyApproval> listStudyApproval = null;
			try {
				listStudyApproval = this.getStudyApprovalsByTitle(studyApprovalTitle);
			} catch (GenisisServiceException e) {
				LOGGER.error("Exception occured while querying getStudyApprovalsByTitle from database", e);
			}
			LdapStudyApprovalGroup lsaGroup = new LdapStudyApprovalGroup();
			lsaGroup.setGroupName(studyApprovalTitle);
			lsaGroup.setLdapStudyApprovals(listStudyApproval);
			ldapStudyApprovalGroups.set(ldapStudyApprovalGroups.indexOf(ldapStudyApprovalGroup), lsaGroup);
		}
	}
}
